<?php
namespace App\Controllers;

use App\Models\Order;
use App\Models\Product;
use Config\Database;

class AnalyticsController {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    public function getAnalytics() {
        // die('test');
        try {
            $response = [
                'summary' => $this->getSummary(),
                'sales_chart' => $this->getSalesChart(),
                'top_products' => $this->getTopProducts(),
                'status_distribution' => $this->getStatusDistribution()
            ];

            $json = json_encode(['status' => true, 'data' => $response]);
            if ($json === false) {
                echo json_encode(['status' => false, 'message' => 'JSON Encode Error: ' . json_last_error_msg()]);
            } else {
                echo $json;
            }
        } catch (\Exception $e) {
            echo json_encode(['status' => false, 'message' => $e->getMessage()]);
        }
    }

    private function getSummary() {
        // Total Sales (Revenue)
        $queryRevenue = "SELECT SUM(total_amount) as total_revenue FROM orders WHERE status != 'cancelled'";
        $stmtRevenue = $this->conn->prepare($queryRevenue);
        $stmtRevenue->execute();
        $revenue = $stmtRevenue->fetch(\PDO::FETCH_ASSOC)['total_revenue'] ?? 0;

        // Total Orders
        $queryOrders = "SELECT COUNT(*) as total_orders FROM orders";
        $stmtOrders = $this->conn->prepare($queryOrders);
        $stmtOrders->execute();
        $orders = $stmtOrders->fetch(\PDO::FETCH_ASSOC)['total_orders'] ?? 0;

        // Total Products
        $queryProducts = "SELECT COUNT(*) as total_products FROM products";
        $stmtProducts = $this->conn->prepare($queryProducts);
        $stmtProducts->execute();
        $products = $stmtProducts->fetch(\PDO::FETCH_ASSOC)['total_products'] ?? 0;

        // Average Order Value
        $avgOrderValue = $orders > 0 ? $revenue / $orders : 0;

        return [
            'total_revenue' => $revenue,
            'total_orders' => $orders,
            'total_products' => $products,
            'avg_order_value' => $avgOrderValue
        ];
    }

    private function getSalesChart() {
        // Last 7 days sales
        $query = "SELECT DATE(created_at) as date, SUM(total_amount) as amount 
                  FROM orders 
                  WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 6 DAY) 
                  AND status != 'cancelled'
                  GROUP BY DATE(created_at) 
                  ORDER BY date ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $results = $stmt->fetchAll(\PDO::FETCH_ASSOC);

        // Fill in missing dates with 0
        $data = [];
        $today = new \DateTime();
        for ($i = 6; $i >= 0; $i--) {
            $date = (clone $today)->modify("-$i days")->format('Y-m-d');
            $amount = 0;
            foreach ($results as $row) {
                if ($row['date'] === $date) {
                    $amount = (float)$row['amount'];
                    break;
                }
            }
            $data[] = ['date' => $date, 'sales' => $amount];
        }

        return $data;
    }

    private function getTopProducts() {
        $query = "SELECT p.name, SUM(oi.quantity) as total_sold, SUM(oi.price * oi.quantity) as revenue
                  FROM order_items oi
                  JOIN products p ON oi.product_id = p.id
                  JOIN orders o ON oi.order_id = o.id
                  WHERE o.status != 'cancelled'
                  GROUP BY p.id, p.name
                  ORDER BY total_sold DESC
                  LIMIT 5";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    private function getStatusDistribution() {
        $query = "SELECT status, COUNT(*) as count FROM orders GROUP BY status";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
}
