<?php
namespace App\Models;

use Config\Database;

class Order {
    private $conn;
    private $table_name = "orders";

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function create($data) {
        try {
            $this->conn->beginTransaction();

            // 1. Insert Order
            $query = "INSERT INTO " . $this->table_name . " 
                    (order_number, customer_name, customer_email, customer_phone, shipping_address, total_amount, status, created_at)
                    VALUES (:order_number, :customer_name, :customer_email, :customer_phone, :shipping_address, :total_amount, 'pending', NOW())";
            
            $stmt = $this->conn->prepare($query);
            
            // Generate unique order number (e.g., ORD-timestamp-random)
            $order_number = 'ORD-' . time() . '-' . rand(100, 999);
            
            $stmt->bindParam(":order_number", $order_number);
            $stmt->bindParam(":customer_name", $data['customer_name']);
            $stmt->bindParam(":customer_email", $data['customer_email']);
            $stmt->bindParam(":customer_phone", $data['customer_phone']);
            $stmt->bindParam(":shipping_address", $data['shipping_address']);
            $stmt->bindParam(":total_amount", $data['total_amount']);
            
            if (!$stmt->execute()) {
                throw new \Exception("Failed to create order record.");
            }
            
            $order_id = $this->conn->lastInsertId();

            // 2. Insert Order Items
            $query_item = "INSERT INTO order_items (order_id, product_id, product_name, quantity, price) VALUES (:order_id, :product_id, :product_name, :quantity, :price)";
            $stmt_item = $this->conn->prepare($query_item);

            foreach ($data['items'] as $item) {
                $stmt_item->bindParam(":order_id", $order_id);
                $stmt_item->bindParam(":product_id", $item['id']);
                $stmt_item->bindParam(":product_name", $item['name']);
                $stmt_item->bindParam(":quantity", $item['quantity']);
                $stmt_item->bindParam(":price", $item['price']);
                
                if (!$stmt_item->execute()) {
                    throw new \Exception("Failed to create order item.");
                }
            }

            $this->conn->commit();
            return ["status" => true, "order_number" => $order_number];

        } catch (\Exception $e) {
            $this->conn->rollBack();
            return ["status" => false, "message" => $e->getMessage()];
        }
    }

    public function getAll() {
        try {
            $query = "SELECT * FROM " . $this->table_name . " ORDER BY created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            
            $orders = [];
            while ($row = $stmt->fetch(\PDO::FETCH_ASSOC)) {
                // Fetch items for this order with product images
                $query_items = "SELECT oi.*, p.image_url 
                               FROM order_items oi 
                               LEFT JOIN products p ON oi.product_id = p.id 
                               WHERE oi.order_id = :order_id";
                $stmt_items = $this->conn->prepare($query_items);
                $stmt_items->bindParam(":order_id", $row['id']);
                $stmt_items->execute();
                $row['items'] = $stmt_items->fetchAll(\PDO::FETCH_ASSOC);
                
                $orders[] = $row;
            }
            
            return $orders;
        } catch (\Exception $e) {
            error_log("Order getAll error: " . $e->getMessage());
            throw $e;
        }
    }

    public function updateStatus($id, $status) {
        $valid_statuses = ['pending', 'processing', 'completed', 'cancelled'];
        if (!in_array($status, $valid_statuses)) {
            return false;
        }

        $query = "UPDATE " . $this->table_name . " SET status = :status WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":id", $id);
        
        return $stmt->execute();
    }

    public function getById($id) {
        try {
            $query = "SELECT * FROM " . $this->table_name . " WHERE id = :id LIMIT 1";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":id", $id);
            $stmt->execute();
            
            $order = $stmt->fetch(\PDO::FETCH_ASSOC);
            
            if ($order) {
                // Fetch items for this order with product images
                $query_items = "SELECT oi.*, p.image_url 
                               FROM order_items oi 
                               LEFT JOIN products p ON oi.product_id = p.id 
                               WHERE oi.order_id = :order_id";
                $stmt_items = $this->conn->prepare($query_items);
                $stmt_items->bindParam(":order_id", $order['id']);
                $stmt_items->execute();
                $order['items'] = $stmt_items->fetchAll(\PDO::FETCH_ASSOC);
            }
            
            return $order;
        } catch (\Exception $e) {
            error_log("Order getById error: " . $e->getMessage());
            throw $e;
        }
    }

    public function delete($id) {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":id", $id);
        return $stmt->execute();
    }
}
